<?php

namespace App\Http\Controllers;

use App\Models\Parametro;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class ParametroController extends Controller
{
    public function index()
    {
        $tipos = Parametro::getTipos();
        return view('parametros.index', compact('tipos'));
    }

    public function getByTipo(Request $request)
    {
        try {
            $tipo = $request->get('tipo');
            
            $parametros = Parametro::porTipo($tipo)
                ->orderBy('nombre')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $parametros
            ]);
        } catch (\Exception $e) {
            Log::error('Error al obtener parámetros: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener los parámetros'
            ], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'tipo' => 'required|string|max:50',
                'nombre' => 'required|string|max:100'
            ]);

            // Verificar si ya existe
            $existe = Parametro::where('tipo', $request->tipo)
                ->where('nombre', $request->nombre)
                ->exists();

            if ($existe) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ya existe un parámetro con ese nombre para este tipo'
                ], 422);
            }

            $parametro = Parametro::create([
                'tipo' => $request->tipo,
                'nombre' => $request->nombre,
                'activo' => true
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Parámetro creado exitosamente',
                'data' => $parametro
            ]);
        } catch (\Exception $e) {
            Log::error('Error al crear parámetro: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error al crear el parámetro'
            ], 500);
        }
    }

    public function show($id)
    {
        try {
            $parametro = Parametro::findOrFail($id);
            return response()->json([
                'success' => true,
                'data' => $parametro
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Parámetro no encontrado'
            ], 404);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $request->validate([
                'tipo' => 'required|string|max:50',
                'nombre' => 'required|string|max:100'
            ]);

            $parametro = Parametro::findOrFail($id);

            // Verificar si ya existe otro registro con el mismo tipo y nombre
            $existe = Parametro::where('tipo', $request->tipo)
                ->where('nombre', $request->nombre)
                ->where('id', '!=', $id)
                ->exists();

            if ($existe) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ya existe un parámetro con ese nombre para este tipo'
                ], 422);
            }

            $parametro->update([
                'tipo' => $request->tipo,
                'nombre' => $request->nombre
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Parámetro actualizado exitosamente',
                'data' => $parametro
            ]);
        } catch (\Exception $e) {
            Log::error('Error al actualizar parámetro: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar el parámetro'
            ], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $parametro = Parametro::findOrFail($id);
            
            // Eliminación lógica
            $parametro->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'Parámetro eliminado exitosamente'
            ]);
        } catch (\Exception $e) {
            Log::error('Error al eliminar parámetro: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar el parámetro'
            ], 500);
        }
    }
}
