<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use OwenIt\Auditing\Contracts\Auditable;
use Wildside\Userstamps\Userstamps;

class Proceso extends Model implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use HasFactory, SoftDeletes, Userstamps;

    protected $fillable = [
        'nuri',
        'codigo_pac',
        'fecha_ingreso',
        'tipo_gasto_id',
        'objeto_contratacion',
        'normativa_id',
        'modalidad_id',
        'precio_referencial',
        'fuente_organismo',
        'numero_certificacion',
        'fecha_autorizacion',
        'nro_resolucion_autorizacion',
        'fecha_resolucion_autorizacion',
        'fecha_carta_adjudicacion',
        'fecha_notificacion_adjudicacion',
        'monto_adjudicado',
        'empresa_adjudicada_id',
        'fecha_remision_juridica',
        'fecha_retorno_juridica',
        'tipo_contrato',
        'fecha_contrato',
        'monto_contrato',
        'empresa_contratada_id',
        'plazo_entrega',
        'fecha_estimada_entrega',
        'cuce',            
        'registro_contraloria',
        'fecha_informe_conformidad',
        'fecha_registro_sicoes',
        'situacion_proceso',
        'fecha_resimision_pago',
        'recurrentes',            
        'tipo_adjunto_id',
        'entidad_id',
    ];

    protected static function booted()
    {
        static::creating(function ($proceso) {
            // Obtener la entidad y modalidad
            $entidad = $proceso->entidad()->first();
            
            $modalidad = $proceso->modalidad()->first();
            $gestion_contrato=date('Y', strtotime($proceso->fecha_ingreso));

            // Calcular el número incremental para esa gestión
            $ultimo = self::where('entidad_id', $proceso->entidad_id)
                ->whereYear('fecha_ingreso', $gestion_contrato)
                ->max('nro');

            $nro = $ultimo ? $ultimo + 1 : 1;
            $proceso->nro = $nro;

            // Formato con ceros: 001, 002, ...
            // $codigo = sprintf(
            //     "SEDEM/%s/%s/%d-%03d",
            //     $entidad->nombre, // Asegúrate que la tabla tenga este campo
            //     $modalidad->codigo, // Igual que arriba
            //     $gestion_contrato,
            //     $nro
            // );
            $codigo = sprintf(
                "SEDEM/PROCESO/%d-%03d",
                $gestion_contrato,
                $nro
            );

            $proceso->codigo = $codigo;
        });
    }
    
    public function entidad()
    {
        return $this->belongsTo(Parametro::class, 'entidad_id');
    }

    public function modalidad()
    {
        return $this->belongsTo(Parametro::class, 'modalidad_id');
    }

    public function normativa()
    {
        return $this->belongsTo(Parametro::class, 'normativa_id');
    }

    public function tipoGasto()
    {
        return $this->belongsTo(Parametro::class, 'tipo_gasto_id');
    }

    public function asignaciones()
    {
        return $this->belongsToMany(User::class, 'asignacions', 'proceso_id', 'responsable_id')
                ->withPivot('fecha_asignacion', 'observacion')
                ->withTimestamps();
    }

    public function ultimaAsignacion()
    {
        return $this->hasOne(Asignacion::class)->latestOfMany(); // Laravel 8+
    }

    public function seguimientos()
    {
        return $this->hasMany(Seguimiento::class);
    }

    public function ultimoSeguimiento()
    {
        return $this->hasOne(Seguimiento::class)->latestOfMany();
    }

    public function empresaAdjudicada()
    {
        return $this->belongsTo(Empresa::class, 'empresa_adjudicada_id');
    }

    public function empresaContratada()
    {
        return $this->belongsTo(Empresa::class, 'empresa_contratada_id');
    }

    public function items()
    {
        return $this->hasMany(ProcesoItem::class);
    }
}
