<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Proceso;
use App\Models\Audiencia;
use App\Models\Parametro;
use Illuminate\Http\Request;
use RealRashid\SweetAlert\Facades\Alert;

class AudienciaController extends Controller
{
    public function index(Request $request)
    {
        $audiencias = Audiencia::with('abogado')
            ->when($request->estado, fn($q) =>
                $q->where('estado', $request->estado)
            )
            ->orderBy('fecha_audiencia', 'desc')
            ->paginate(10);

        return view('audiencias.index', compact('audiencias'));
    }

    public function create()
    {
        $abogados = User::orderBy('name')->get();
        $tipo_audiencias = Parametro::where('tipo', 'AUDIENCIA')->get();
        $estados = Parametro::where('tipo', 'ESTADO_AUDIENCIA')->get();
        $procesos = Proceso::where('estado', true)->get();
        return view('audiencias.create', compact('abogados', 'tipo_audiencias', 'estados', 'procesos'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'proceso_id'     => 'required',
            'departamento'    => 'required',
            'tipo_audiencia_id'  => 'required',
            'fecha_audiencia' => 'required|date',
            'hora_audiencia'  => 'required',
            'lugar'           => 'required',
            'estado'      => 'required',
            'abogado_id'      => 'required|array',
            'abogado_id.*'    => 'exists:users,id',
            'juzgado'        => 'required',
            'sala'           => 'required',
        ]);

        $duplicado = Audiencia::where('proceso_id', $request->proceso_id)
            ->where('fecha_audiencia', $request->fecha_audiencia)
            ->where('hora_audiencia', $request->hora_audiencia)
            ->exists();

        if ($duplicado) {
            Alert::error('Error', 'La audiencia ya está registrada para esa fecha y hora.');
            return back()->withInput();
        }
        
        Audiencia::create([
            'proceso_id'     => $request->proceso_id,
            'departamento'    => $request->departamento,
            'tipo_audiencia_id'  => $request->tipo_audiencia_id,
            'fecha_audiencia' => $request->fecha_audiencia,
            'hora_audiencia'  => $request->hora_audiencia,
            'lugar'           => $request->lugar,
            'representantes'      => implode(';', $request->abogado_id), 
            'estado_id'      => $request->estado,
            'juzgado'        => $request->juzgado,
            'sala'           => $request->sala,
        ]);

        Alert::success('Éxito', 'Audiencia registrada correctamente!');
        
        return redirect()
            ->route('audiencias.index')
            ->with('success', 'Audiencia registrada correctamente');
    }

    public function show(Audiencia $audiencia)
    {
        $abogados = User::orderBy('name')->get();
        $tipo_audiencias = Parametro::where('tipo', 'AUDIENCIA')->get();
        $estados = Parametro::where('tipo', 'ESTADO_AUDIENCIA')->get();
        $procesos = Proceso::all();
        return view('audiencias.show', compact('audiencia'));
    }

    public function edit(Audiencia $audiencia)
    {
        $abogados = User::orderBy('name')->get();
        $tipo_audiencias = Parametro::where('tipo', 'AUDIENCIA')->get();
        $estados = Parametro::where('tipo', 'ESTADO_AUDIENCIA')->get();
        $procesos = Proceso::where('estado', true)->get();
        return view('audiencias.edit', compact('audiencia','abogados','tipo_audiencias','estados','procesos'));
    }

    public function update(Request $request, Audiencia $audiencia)
    {
        $request->validate([
            'proceso_id'     => 'required',
            'departamento'    => 'required',
            'tipo_audiencia_id'  => 'required',
            'fecha_audiencia' => 'required|date',
            'hora_audiencia'  => 'required',
            'lugar'           => 'required',
            'estado'      => 'required',
            'abogado_id'      => 'required|array',
            'abogado_id.*'    => 'exists:users,id',
            'juzgado'        => 'required',
            'sala'           => 'required',
        ]);

        $audiencia->update([
            'proceso_id'     => $request->proceso_id,
            'departamento'    => $request->departamento,
            'tipo_audiencia_id'  => $request->tipo_audiencia_id,
            'fecha_audiencia' => $request->fecha_audiencia,
            'hora_audiencia'  => $request->hora_audiencia,
            'lugar'           => $request->lugar,
            'representantes'      => implode(';', $request->abogado_id), 
            'estado_id'      => $request->estado,
            'juzgado'        => $request->juzgado,
            'sala'           => $request->sala,
        ]);

        Alert::success('Éxito', 'Audiencia actualizada correctamente!');
        return redirect()->route('audiencias.index');
    }

    public function destroy(Audiencia $audiencia)
    {
        $audiencia->delete();

        Alert::success('Éxito', 'Audiencia eliminada');
        return redirect()->route('audiencias.index');
    }
}
