<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use YoHang88\LetterAvatar\LetterAvatar;
use RealRashid\SweetAlert\Facades\Alert;
use Illuminate\Validation\Rules\Password;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');

    }
    public function index()
    {
        $users=User::all();
        return view('admin.users.index',compact('users'));
    }

    public function create()
    {
        //$empleadosRRHH=ServiceEmpleadosRRHH::all();
        $roles=Role::where('id','>',1)->get();
        $user=new User();
        return view('admin.users.create',compact('roles','user'));
    }

    public function store(Request $request)
    {
        $user=new User();

        $user->name = $request->name;
        $user->username = $request->username;
        $user->email = $request->email;
        //datos desde RRHH

        $user->password = Hash::make($request->password);
        $user->save();

        $user->roles()->sync($request->role_id);

        Alert::success("Usuario registrado correctamente!");
        return redirect()->route('users.index');
    }

    
    public function edit(User $user)
    {
        
        $roles=Role::where('id','>',1)->get();
        
        return view('admin.users.edit',compact('user','roles'));
    }

    public function update(Request $request, User $user)
    {
        $user->name = $request->name;
        $user->username = $request->username;
        $user->email = $request->email;

        if(isset($request->password)){
            $user->password = Hash::make($request->password);
        }
        
        $user->save();
        //actualice los roles
        $user->roles()->sync($request->role_id);
        
        Alert::success('Datos actualizados correctamente!');
        return redirect()->route('users.index');
    }

    public function update_profile(Request $request, User $user){
        $request->validate([
            'name' => 'required',
            'avatar' => 'image|mimes:jpeg,png,jpg,gif,svg',
            'username' => [
                'required',
                'string',
                'max:255',
                Rule::unique('users')->ignore($user->id),
            ],
        ],[
            'username.unique' => 'El nombre de usuario ya existe',
            'username.required' => 'El nombre de usuario es obligatorio',
            'name.required' => 'El nombre es obligatorio',
        ]);

        $nombre_usuario=$user->username;
        $user->name=$request->name;
        $user->username=$request->username;   
        
        $file_foto=$request->file('avatar');
        if($file_foto != null){
            $direccion=public_path()."/assets/images/auth/";
            $fullPath=public_path().'/assets/images/auth/'.$user->avatar;
            File::delete($fullPath);
            $filename=$file_foto->getClientOriginalName();
            $file_foto->move($direccion, $filename);
            $user->avatar=$filename;
        }else{
            $path=public_path()."/assets/images/auth/".$user->id.'.jpg';
            if (!file_exists($path)){
                $avatar = new LetterAvatar($request->name,'circle', 64);
                $avatar->saveAs($path, LetterAvatar::MIME_TYPE_JPEG);
            }
            $user->avatar=$user->id.'.jpg';
        }
        $user->save();
        if($nombre_usuario != $user->username){
            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();
            Alert::success('Su Nombre de Usuario ha sido actualizado!', 'Por favor, ingrese nuevamente con su nuevo nombre de usuario.');
            return redirect()->route('login');
        }
        Alert::success('Datos actualizados correctamente!');
        return redirect()->back();
    }

    public function changePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'new_password' => [
                'required',
                'confirmed',
                Password::min(8) // Debe tener por lo menos 8 caracteres
                    ->mixedCase() // Debe tener mayúsculas + minúsculas
                    ->letters() // Debe incluir letras
                    ->numbers() // Debe incluir números
                    ->symbols(), // Debe incluir símbolos
                ],
        ],[
            'new_password.confirmed' => 'La nueva contraseña no coincide',
            'new_password.min' => 'La nueva contraseña debe tener al menos 8 caracteres',
            'current_password.required' => 'La contraseña actual es obligatoria',
            'new_password.required' => 'La nueva contraseña es obligatoria',
            'password' => 'La nueva contraseña no coincide con la confirmación',
            'password.mixed' => 'La nueva contraseña debe tener mayúsculas y minúsculas',
            'password.letters' => 'La nueva contraseña debe tener letras',
            'password.numbers' => 'La nueva contraseña debe tener números',
            'password.symbols' => 'La nueva contraseña debe tener símbolos',
        ]);

        $user=User::find(auth()->user()->id);

        // Logging para depuración
        Log::info('Intento de cambio de contraseña para el usuario: ' . $user->id);

        if (!Hash::check($request->current_password, $user->password)) {
            Log::warning('Contraseña actual incorrecta para el usuario: ' . $user->id);
            Alert::error('Error', 'La contraseña actual no es correcta');
            return redirect()->back()->withErrors(['current_password' => 'La contraseña actual no es correcta']);
        }

        // Actualizar la contraseña
        $user->password = Hash::make($request->new_password);
        
        // Intentar guardar y verificar si fue exitoso
        if ($user->save()) {
            Log::info('Contraseña actualizada con éxito para el usuario: ' . $user->id);
            
            // Verificar si la nueva contraseña se guardó correctamente
            if (Hash::check($request->new_password, $user->fresh()->password)) {
                // Cerrar sesión y redirigir
                Auth::logout();
                $request->session()->invalidate();
                $request->session()->regenerateToken();
                Alert::success('Contraseña ha sido actualizada!', 'Por favor, ingrese nuevamente con su nueva contraseña.');
                return redirect()->route('login');
            } else {
                Log::error('La nueva contraseña no se guardó correctamente para el usuario: ' . $user->id);
                Alert::error('Error', 'Hubo un problema al guardar la nueva contraseña. Por favor, inténtelo de nuevo.');
                return redirect()->back();
            }
        } else {
            Log::error('No se pudo guardar la nueva contraseña para el usuario: ' . $user->id);
            Alert::error('Error', 'No se pudo actualizar la contraseña. Por favor, inténtelo de nuevo.');
            return redirect()->back();
        }
        
    }
    public function destroy(User $user)
    {
        $user->delete();
        Alert::success('Usuario Eliminado correctamente!');
        return redirect()->route('users.index');
    }

    public function update_email(Request $request){
        $request->validate([
            'email' => 'required',
        ],[
            'email.required' => 'El correo es obligatorio',
        ]);
        $user=User::find(auth()->user()->id);
        $user->email=$request->email;
        $user->email_verified_at = date('Y-m-d H:i:s');
        $user->save();

        Alert::success('Correo actualizado correctamente!');

        return redirect()->back();
    }
}
